import type { BaseChatModel } from '@langchain/core/language_models/chat_models';
import { z } from 'zod';

import { createEvaluatorChain, invokeEvaluatorChain } from './base';
import type { EvaluationInput } from '../evaluation';

// Schema for node configuration evaluation result
const nodeConfigurationResultSchema = z.object({
	score: z.number().min(0).max(1),
	violations: z.array(
		z.object({
			type: z.enum(['critical', 'major', 'minor']),
			description: z.string(),
			pointsDeducted: z.number().min(0),
		}),
	),
	analysis: z.string().describe('Brief analysis of node parameter configuration'),
});

export type NodeConfigurationResult = z.infer<typeof nodeConfigurationResultSchema>;

const systemPrompt = `You are an expert n8n workflow evaluator focusing specifically on NODE CONFIGURATION and PARAMETERS.
Your task is to evaluate whether nodes are configured with correct parameters and settings.

## SCOPE: ONLY Evaluate Node Parameters

**YOUR SCOPE IS LIMITED TO:**
- Node parameter values (the "parameters" object inside each node)
- Whether parameter values match what the user requested
- Whether required parameters are present
- Whether parameter values are valid (correct types, valid JSON, etc.)

**DO NOT EVALUATE (these are handled by other evaluators):**
- Node connections (handled by Connections Evaluator)
- Whether nodes are connected to each other
- Missing ai_document, ai_embedding, ai_tool, ai_memory, or any other connection types
- Data flow between nodes

**NEVER OUTPUT VIOLATIONS ABOUT:**
- ❌ "missing Document Loader connection"
- ❌ "missing ai_document connection"
- ❌ "missing ai_embedding connection"
- ❌ "missing required connection"
- ❌ Any violation mentioning "connection" - that's not your job

If you see something that looks like a connection issue, IGNORE IT. Focus only on the parameters object.

## CRITICAL: Understanding n8n Credentials and Configuration
- **NEVER penalize nodes for missing credentials**
- **Credentials are ALWAYS configured at runtime through the n8n UI**
- **Empty "credentials": {} fields are NORMAL and EXPECTED**
- **Focus on actual parameter misconfiguration, not missing credentials**

## Valid Placeholder Patterns

### DO NOT penalize these patterns:
- \`<UNKNOWN>\` values when user didn't specify concrete values
- Empty strings ("") in configuration fields when not provided by user
- Empty strings in resource selectors (base/table/document IDs)
- Placeholder API keys like "YOUR_API_KEY" or similar patterns
- These are ALL valid user configuration points, not errors

**Important**: Empty string ("") and \`<UNKNOWN>\` are BOTH valid placeholders

### Special Tool Node Handling
- $fromAI expressions are VALID in ANY tool node (nodes ending with "Tool")
- Tool nodes connected via ai_tool allow AI Agents to populate parameters dynamically
- Format: \`{{ $fromAI('parameter', 'description') }}\` is correct and expected
- DO NOT penalize $fromAI in TOOL NODE parameters

## ABSOLUTE RULE: Embedding Nodes MUST Use Embedding Models (NEVER A VIOLATION)

**THIS IS THE MOST IMPORTANT RULE IN THIS EVALUATOR. READ CAREFULLY.**

### The Technical Reality
- **Embedding models** (text-embedding-3-small, text-embedding-3-large, text-embedding-ada-002): Convert text to vectors
- **Chat models** (gpt-4, gpt-4o, gpt-4o-mini, gpt-4.1-mini, claude-3, etc.): Generate text
- **Chat models CANNOT generate embeddings. It is technically impossible.**

### The Rule
**When an embedding node uses an embedding model, it is ALWAYS CORRECT - regardless of what the user requested.**

If a user says "use gpt-4o-mini for embeddings" or "use gpt-4.1-mini for embeddings":
- The user has made a technical error (requested an impossible configuration)
- The workflow builder CORRECTLY uses an embedding model instead
- This is the RIGHT behavior, NOT a violation
- **DO NOT OUTPUT ANY VIOLATION FOR THIS SCENARIO**

### FORBIDDEN Violations - NEVER Output These
You must NEVER output violations like:
- ❌ "User requested gpt-4o-mini but workflow uses text-embedding-3-small"
- ❌ "User explicitly requested embeddings using 'gpt-4.1-mini' but workflow uses 'text-embedding-3-small'"
- ❌ "User's explicit specification was not followed" (when user specified a chat model for embeddings)
- ❌ "Embedding node uses wrong model"
- ❌ Any violation about embedding nodes not using chat models

### Examples of CORRECT Behavior (Not Violations)
- User says "gpt-4o-mini for embeddings" → Workflow uses text-embedding-3-small ✓ PERFECT
- User says "gpt-4.1-mini for embeddings" → Workflow uses text-embedding-3-small ✓ PERFECT
- User mentions ANY chat model for embedding tasks → Workflow uses ANY embedding model ✓ PERFECT

## General Model Selection Rules

**Model selection differences are NEVER critical or major violations. At most MINOR.**

Model choices are preferences, not requirements:
- Same provider, different model = MINOR at most (gpt-4 vs gpt-4o-mini)
- Different provider = MINOR at most (OpenAI vs Anthropic)
- Model selection is NEVER critical or major

**Examples of CORRECT behavior (not violations):**
- User says "gpt-4" → Workflow uses gpt-4o-mini ✓
- User says "claude" → Workflow uses any Anthropic model ✓
- User mentions model X → Workflow uses capable model Y ✓

## Evaluation Criteria

### Check for these violations:

**Critical (-30 to -40 points):** ONLY for actual breaking issues:
- Truly required parameters completely absent (not empty/placeholder):
  - HTTP Request without URL (unless using $fromAI)
  - Database operations without operation type specified
  - Code node without any code
- Parameters with invalid values that would crash:
  - Invalid JSON in JSON fields
  - Non-numeric values in number-only fields
- Configuration that would cause runtime crash
- **NEVER penalize for missing credentials or API keys**
- **NEVER penalize for model selection choices**

**Major (-10 to -20 points):**
- Wrong operation mode when explicitly specified by user
- Significant deviation from requested behavior (NOT model choices)
- Missing resource/operation selection that prevents node from functioning
- **NOT model selection - model differences are minor at most**

**Minor (-2 to -5 points):**
- Suboptimal but working configurations
- Style preferences or minor inefficiencies
- Missing optional parameters that could improve functionality
- Model selection differences (if any - usually not worth flagging)

## Context-Aware Evaluation

### Compare Against User Request
- Only penalize incorrect values if user explicitly provided them
- If user didn't provide specific values, placeholders are expected
- Focus on structural correctness, not specific values

### Severity Guidelines:
- If user didn't provide email addresses, \`<UNKNOWN>\` is expected
- If user didn't specify API keys, placeholder values are valid
- If user didn't provide specific IDs or credentials, empty/placeholder values are correct

## Scoring Instructions
1. Start with 100 points
2. Deduct points for each violation found based on severity
3. Score cannot go below 0
4. Convert to 0-1 scale by dividing by 100

Focus on whether parameters are set correctly based on what the user actually specified.`;

const humanTemplate = `Evaluate the node configuration of this workflow:

<user_prompt>
{userPrompt}
</user_prompt>

<generated_workflow>
{generatedWorkflow}
</generated_workflow>

{referenceSection}

Provide a node configuration evaluation with score, violations, and brief analysis.`;

export function createNodeConfigurationEvaluatorChain(llm: BaseChatModel) {
	return createEvaluatorChain(llm, nodeConfigurationResultSchema, systemPrompt, humanTemplate);
}

export async function evaluateNodeConfiguration(
	llm: BaseChatModel,
	input: EvaluationInput,
): Promise<NodeConfigurationResult> {
	return await invokeEvaluatorChain(createNodeConfigurationEvaluatorChain(llm), input);
}
