import type { ChatHubLLMProvider } from '@n8n/api-types';

/* eslint-disable @typescript-eslint/naming-convention */

// Maximum context window tokens for various models across supported providers.
// Based on a snapshot of model capabilities as of October 2025, with a list of models
// that regular OpenAI / Anthropic / Google API credentials currently have access to at n8n.
// If the limit is set to 0, it means either the model has no defined limit or the information
// is not availabl and no context window trimming is applied. Similarly, if the model used is
// not listed, no limit is applied.
export const maxContextWindowTokens: Record<ChatHubLLMProvider, Record<string, number>> = {
	openai: {
		'chatgpt-4o-latest': 128000,
		'codex-mini-latest': 0,
		'gpt-3.5-turbo': 16392,
		'gpt-3.5-turbo-0125': 16392,
		'gpt-3.5-turbo-1106': 16392,
		'gpt-3.5-turbo-16k': 16392,
		'gpt-4': 8196,
		'gpt-4-0125-preview': 8196,
		'gpt-4-0613': 8196,
		'gpt-4-1106-preview': 8196,
		'gpt-4-turbo': 8196,
		'gpt-4-turbo-2024-04-09': 8196,
		'gpt-4-turbo-preview': 8196,
		'gpt-4.1': 1048576,
		'gpt-4.1-2025-04-14': 1048576,
		'gpt-4.1-mini': 1048576,
		'gpt-4.1-mini-2025-04-14': 1048576,
		'gpt-4.1-nano': 1048576,
		'gpt-4.1-nano-2025-04-14': 1048576,
		'gpt-4o': 128000,
		'gpt-4o-2024-05-13': 128000,
		'gpt-4o-2024-08-06': 128000,
		'gpt-4o-2024-11-20': 128000,
		'gpt-4o-audio-preview': 128000,
		'gpt-4o-audio-preview-2024-10-01': 128000,
		'gpt-4o-audio-preview-2024-12-17': 128000,
		'gpt-4o-audio-preview-2025-06-03': 128000,
		'gpt-4o-mini': 128000,
		'gpt-4o-mini-2024-07-18': 128000,
		'gpt-4o-mini-audio-preview': 128000,
		'gpt-4o-mini-audio-preview-2024-12-17': 128000,
		'gpt-4o-mini-search-preview': 128000,
		'gpt-4o-mini-search-preview-2025-03-11': 128000,
		'gpt-4o-mini-transcribe': 128000,
		'gpt-4o-search-preview': 128000,
		'gpt-4o-search-preview-2025-03-11': 128000,
		'gpt-4o-transcribe': 128000,
		'gpt-4o-transcribe-diarize': 128000,
		'gpt-5': 400000,
		'gpt-5-2025-08-07': 400000,
		'gpt-5-chat-latest': 128000,
		'gpt-5-codex': 400000,
		'gpt-5-mini': 400000,
		'gpt-5-mini-2025-08-07': 400000,
		'gpt-5-nano': 400000,
		'gpt-5-nano-2025-08-07': 400000,
		'gpt-5-pro': 196000,
		'gpt-5-pro-2025-10-06': 196000,
		'gpt-5-search-api': 0,
		'gpt-5-search-api-2025-10-14': 0,
		'gpt-5.1': 400000,
		'gpt-5.1-2025-11-13': 400000,
		'gpt-5.1-chat-latest': 128000,
		'gpt-5.1-codex': 400000,
		'gpt-5.1-codex-mini': 400000,
		'gpt-5.2': 400000,
		'gpt-5.2-2025-12-11': 400000,
		'gpt-5.2-chat-latest': 400000,
		'gpt-5.2-pro': 400000,
		'gpt-5.2-pro-2025-12-11': 400000,
		'gpt-audio': 0,
		'gpt-audio-2025-08-28': 0,
		'gpt-audio-mini': 0,
		'gpt-audio-mini-2025-10-06': 0,
		'gpt-image-1': 0,
		'gpt-image-1-mini': 0,
		o1: 200000,
		'o1-2024-12-17': 200000,
		'o1-mini': 200000,
		'o1-mini-2024-09-12': 200000,
		'o1-pro': 200000,
		'o1-pro-2025-03-19': 200000,
		o3: 200000,
		'o3-2025-04-16': 200000,
		'o3-mini': 200000,
		'o3-mini-2025-01-31': 200000,
		'o4-mini': 200000,
		'o4-mini-2025-04-16': 200000,
		'o4-mini-deep-research': 0,
		'o4-mini-deep-research-2025-06-26': 0,
	},
	anthropic: {
		'claude-opus-4-5-20251101': 200000,
		'claude-haiku-4-5-20251001': 200000,
		'claude-sonnet-4-5-20250929': 200000,
		'claude-opus-4-1-20250805': 200000,
		'claude-opus-4-20250514': 200000,
		'claude-sonnet-4-20250514': 200000,
		'claude-3-7-sonnet-20250219': 200000,
		'claude-3-5-haiku-20241022': 200000,
		'claude-3-haiku-20240307': 200000,
	},
	google: {
		'models/aqa': 0,
		'models/gemini-2.0-flash': 1048576,
		'models/gemini-2.0-flash-001': 1048576,
		'models/gemini-2.0-flash-exp': 1048576,
		'models/gemini-2.0-flash-lite': 1048576,
		'models/gemini-2.0-flash-lite-001': 1048576,
		'models/gemini-2.0-flash-lite-preview': 1048576,
		'models/gemini-2.0-flash-lite-preview-02-05': 1048576,
		'models/gemini-2.0-flash-thinking-exp': 32000,
		'models/gemini-2.0-flash-thinking-exp-01-21': 32000,
		'models/gemini-2.0-flash-thinking-exp-1219': 32000,
		'models/gemini-2.0-pro-exp': 2000000,
		'models/gemini-2.0-pro-exp-02-05': 2000000,
		'models/gemini-2.5-computer-use-preview-10-2025': 0,
		'models/gemini-2.5-flash': 1048576,
		'models/gemini-2.5-flash-image': 1048576,
		'models/gemini-2.5-flash-image-preview': 1048576,
		'models/gemini-2.5-flash-lite': 1048576,
		'models/gemini-2.5-flash-lite-preview-06-17': 1048576,
		'models/gemini-2.5-flash-lite-preview-09-2025': 1048576,
		'models/gemini-2.5-flash-preview-05-20': 1048576,
		'models/gemini-2.5-flash-preview-09-2025': 1048576,
		'models/gemini-2.5-flash-preview-tts': 8000,
		'models/gemini-2.5-pro': 1048576,
		'models/gemini-2.5-pro-preview-03-25': 1048576,
		'models/gemini-2.5-pro-preview-05-06': 1048576,
		'models/gemini-2.5-pro-preview-06-05': 1048576,
		'models/gemini-2.5-pro-preview-tts': 8000,
		'models/gemini-3-pro-preview': 1048576,
		'models/gemini-3-pro-image-preview': 65536,
		'models/gemini-exp-1206': 2000000,
		'models/gemini-flash-latest': 1048576,
		'models/gemini-flash-lite-latest': 1048576,
		'models/gemini-pro-latest': 1048576,
		'models/gemini-robotics-er-1.5-preview': 0,
		'models/gemma-3-12b-it': 128000,
		'models/gemma-3-1b-it': 32000,
		'models/gemma-3-27b-it': 128000,
		'models/gemma-3-4b-it': 128000,
		'models/gemma-3n-e2b-it': 32000,
		'models/gemma-3n-e4b-it': 32000,
		'models/imagen-3.0-generate-002': 480,
		'models/imagen-4.0-generate-001': 480,
		'models/imagen-4.0-generate-preview-06-06': 480,
		'models/imagen-4.0-ultra-generate-preview-06-06': 480,
		'models/learnlm-2.0-flash-experimental': 0,
	},
	azureOpenAi: {},
	azureEntraId: {},
	ollama: {},
	awsBedrock: {},
	vercelAiGateway: {},
	xAiGrok: {
		'grok-2-1212': 32768,
		'grok-2-vision-1212': 32768,
		'grok-3': 131072,
		'grok-3-mini': 131072,
		'grok-4-0709': 256000,
		'grok-4-1-fast-non-reasoning': 2000000,
		'grok-4-1-fast-reasoning': 2000000,
		'grok-4-fast-non-reasoning': 2000000,
		'grok-4-fast-reasoning': 2000000,
		'grok-code-fast-1': 256000,
	},
	groq: {
		'qwen/qwen3-32b': 131072,
		'llama-3.3-70b-versatile': 131072,
		'llama-3.1-8b-instant': 131072,
		'playai-tts': 8192,
		'playai-tts-arabic': 8192,
		'openai/gpt-oss-safeguard-20b': 131072,
		'openai/gpt-oss-120b': 131072,
		'openai/gpt-oss-20b': 131072,
		'meta-llama/llama-prompt-guard-2-86m': 512,
		'meta-llama/llama-prompt-guard-2-22m': 512,
		'meta-llama/llama-guard-4-12b': 131072,
		'meta-llama/llama-4-scout-17b-16e-instruct': 131072,
		'meta-llama/llama-4-maverick-17b-128e-instruct': 131072,
		'moonshotai/kimi-k2-instruct-0905': 262144,
		'moonshotai/kimi-k2-instruct': 262144,
		'groq/compound': 131072,
		'groq/compound-mini': 131072,
	},
	openRouter: {},
	deepSeek: {
		'deepseek-chat': 128000,
		'deepseek-reasoner': 128000,
	},
	cohere: {},
	mistralCloud: {
		'mistral-large-2411': 256000,
		'mistral-large-2512': 256000,
		'mistral-large-latest': 256000,
		'mistral-large-pixtral-2411': 256000,
		'mistral-medium': 128000,
		'mistral-medium-2505': 128000,
		'mistral-medium-2508': 128000,
		'mistral-medium-latest': 128000,
		'mistral-small-2506': 128000,
		'mistral-small-latest': 128000,
	},
};

const CONTEXT_WINDOW_SAFETY_FACTOR = 0.95;

export const getMaxContextWindowTokens = (
	provider: ChatHubLLMProvider,
	model: string,
): number | undefined => {
	const limit = maxContextWindowTokens[provider]?.[model] ?? 0;
	if (!limit) {
		return undefined;
	}

	return Math.floor(limit * CONTEXT_WINDOW_SAFETY_FACTOR);
};
