import {
	createTestNode,
	createTestTaskData,
	createTestWorkflowObject,
	defaultNodeDescriptions,
} from '@/__tests__/mocks';
import { createTestLogEntry } from '../__test__/mocks';
import { fireEvent, render, waitFor } from '@testing-library/vue';
import userEvent from '@testing-library/user-event';
import LogsViewRunData from './LogsViewRunData.vue';
import { createTestingPinia, type TestingPinia } from '@pinia/testing';
import { createRunExecutionData, NodeApiError, NodeConnectionTypes } from 'n8n-workflow';
import { AGENT_NODE_TYPE, OPEN_AI_CHAT_MODEL_NODE_TYPE } from '@/app/constants';
import { useNodeTypesStore } from '@/app/stores/nodeTypes.store';
import { mockedStore } from '@/__tests__/utils';

describe('LogsViewRunData', () => {
	let pinia: TestingPinia;

	const nodeB = createTestNode({ name: 'B' });
	const runDataB = createTestTaskData({
		data: {
			main: [
				[{ json: { p: '0' } }, { json: { p: '1' } }, { json: { p: '2' } }, { json: { p: '3' } }],
			],
		},
	});
	const workflow = createTestWorkflowObject({ nodes: [nodeB] });
	const logEntry = createTestLogEntry({
		node: nodeB,
		runIndex: 0,
		runData: runDataB,
		workflow,
		execution: createRunExecutionData({ resultData: { runData: { B: [runDataB] } } }),
	});

	beforeEach(() => {
		pinia = createTestingPinia({ stubActions: false, fakeApp: true });
		const nodeTypesStore = mockedStore(useNodeTypesStore);
		nodeTypesStore.setNodeTypes(defaultNodeDescriptions);
	});

	it('should display item count', async () => {
		const rendered = render(LogsViewRunData, {
			global: { plugins: [pinia] },
			props: { title: '', logEntry, collapsingTableColumnName: null, paneType: 'output' },
		});

		expect(rendered.getByTestId('run-data-item-count')).toHaveTextContent('4 items');
	});

	it('should display matched and total item count unless display mode is schema', async () => {
		const rendered = render(LogsViewRunData, {
			global: { plugins: [pinia] },
			props: { title: '', logEntry, collapsingTableColumnName: null, paneType: 'output' },
		});

		await fireEvent.click(await rendered.findByTestId('radio-button-table'));
		await userEvent.type(await rendered.findByTestId('ndv-search'), '0');

		await waitFor(() => {
			expect(rendered.getByTestId('run-data-item-count')).toHaveTextContent('1 of 4 items');
		});

		await fireEvent.click(await rendered.findByTestId('radio-button-schema'));

		await waitFor(() => {
			expect(rendered.getByTestId('run-data-item-count')).toHaveTextContent('4 items');
		});
	});

	it('should display input data even when the root node finished with an error', async () => {
		const agentNode = createTestNode({ name: 'AI agent', type: AGENT_NODE_TYPE });
		const modelNode = createTestNode({ name: 'AI model', type: OPEN_AI_CHAT_MODEL_NODE_TYPE });

		const aiWorkflow = createTestWorkflowObject({
			nodes: [agentNode, modelNode],
			connections: {
				'AI model': {
					[NodeConnectionTypes.AiLanguageModel]: [
						[{ index: 0, node: 'AI agent', type: NodeConnectionTypes.AiLanguageModel }],
					],
				},
			},
		});
		const modelTaskData = createTestTaskData({
			executionStatus: 'success',
			data: { [NodeConnectionTypes.AiLanguageModel]: [[{ json: { message: 'hello' } }]] },
			source: [{ previousNode: 'AI agent' }],
		});
		const agentTaskData = createTestTaskData({
			executionStatus: 'error',
			error: new NodeApiError(agentNode, {}),
		});
		const aiExecution = createRunExecutionData({
			resultData: {
				runData: {
					'AI agent': [agentTaskData],
					'AI model': [modelTaskData],
				},
			},
		});
		const rendered = render(LogsViewRunData, {
			global: { plugins: [pinia] },
			props: {
				title: '',
				logEntry: createTestLogEntry({
					node: modelNode,
					runIndex: 0,
					runData: modelTaskData,
					workflow: aiWorkflow,
					execution: aiExecution,
					parent: createTestLogEntry({
						node: agentNode,
						runIndex: 0,
						runData: agentTaskData,
						workflow: aiWorkflow,
						execution: aiExecution,
					}),
				}),
				collapsingTableColumnName: null,
				paneType: 'input',
			},
		});

		await rendered.findByText(/hello/);
	});
});
